import matplotlib.pyplot as mpl


Fibonacci_Phi = (5.0 ** 0.5 - 1.0) / 2.0 + 1.0


def log_levels(integer_levels = 3, sub_integral_levels = 1, base = Fibonacci_Phi):
  """Logarithmic equidistant levels

  Parameters:
    integer_levels      : integral steps of exponent levels
                          the same amount is used downside ("retracements")
                          and upside ("extensions")
    sub_integral_levels : steps between two integral exponents for further refinement
    base                : base for exponentiation for log equidistant levels
                          default = Fibonacci number Fibonacci_Phi = 1.61803399
  """

  assert(type(integer_levels     ) == int and integer_levels      > 0)
  assert(type(sub_integral_levels) == int and sub_integral_levels > 0)

  steps = sub_integral_levels * integer_levels * 2 + 1
  return mpl.np.logspace(- integer_levels, integer_levels, steps, True, base)


def music_interval_levels(integer_levels = 3):
  """Levels with rational ratios as in music intervals

  Parameters:
    integer_levels      : integral steps of exponent levels
                          the same amount is used downside ("retracements")
                          and upside ("extensions")
  """

  assert(type(integer_levels) == int and integer_levels > 0)

  intervals = [
    ('Unison'       ,  1,  1)
  , ('Minor Second' , 16, 15)
  , ('Major Second' ,  9,  8)
  , ('Minor Third'  ,  6,  5)
  , ('Major Third'  ,  5,  4)
  , ('Fourth'       ,  4,  3)
  , ('Tritone'      ,  7,  5)
  , ('Fifth'        ,  3,  2)
  , ('Minor Sixth'  ,  8,  5)
  , ('Major Sixth'  ,  5,  3)
  , ('Minor Seventh',  7,  4)
  , ('Major Seventh', 15,  8)
  , ('Octave'       ,  2,  1)
  ]

  levels = [2 ** - integer_levels]
  for octave in range(- integer_levels, integer_levels):
    base_level = levels[-1]
    for interval in intervals[1:]:
      levels.append(base_level * float(interval[1]) / float(interval[2]))
  return levels


def tabulate_log_levels(integer_levels = 2, max_sub_integral_levels = 12, base = Fibonacci_Phi):
  """Tabulate logarithmic equidistant levels"""

  for sub_integral_levels in range(1, max_sub_integral_levels + 1):
    levels = log_levels(integer_levels, sub_integral_levels, base)
    print sub_integral_levels, 'sub levels:',
    for level in levels:
      print round(100 * level, 1),
    print '\n'


def draw_log_levels(integer_levels = 2, max_sub_integral_levels = 12, log = False, base = Fibonacci_Phi, marker = '+'):
  """Draw logarithmic equidistant levels"""

  mpl.title('Many more "Fibonacci levels"')
  mpl.xlabel('Sub integral exponent levels')
  mpl.ylabel('"Fibonacci level"')

  for sub_integral_levels in range(1, max_sub_integral_levels + 1):
    levels = log_levels(integer_levels, sub_integral_levels, base)
    mpl.plot([sub_integral_levels] * len(levels), levels, marker)

  free_space_for_nicer_picture = 1.2
  mpl.axis([0, max_sub_integral_levels + 1, 0, levels[-1] * free_space_for_nicer_picture])
  if log:
    mpl.gca().set_yscale("log")
  mpl.show()


def draw_music_interval_levels(integer_levels = 2, log = False, marker = '+'):
  """Draw logarithmic equidistant levels"""

  mpl.title('Music like intervals')
  mpl.xlabel('')
  mpl.ylabel('Interval level')

  levels = music_interval_levels(integer_levels)
  mpl.plot([1] * len(levels), levels, marker)

  free_space_for_nicer_picture = 1.2
  mpl.axis([0, 2, 0, levels[-1] * free_space_for_nicer_picture])
  mpl.xticks(mpl.np.arange(0))
  if log:
    mpl.gca().set_yscale("log")
  mpl.show()


draw_log_levels(1)
draw_music_interval_levels(1)
